document.addEventListener('DOMContentLoaded', () => {
    console.log("=== DÉMARRAGE DU SCRIPT GEOLOC.JS ===");
    
    // Récupérer l'ID du module depuis les variables JavaScript
    const moduleId = window.moduleId || '';
    console.log("Module ID:", moduleId);
    
    // Sélectionner les éléments avec l'ID du module
    const geolocateButton = document.getElementById('geolocate-button-' + moduleId);
    const distancesInfo = document.getElementById('distances-info-' + moduleId);
    
    // Vérification des éléments DOM
    console.log("Bouton de géolocalisation:", geolocateButton ? "TROUVÉ" : "MANQUANT");
    console.log("Info distances:", distancesInfo ? "TROUVÉ" : "MANQUANT");
    
    if (!geolocateButton) {
        console.error("Bouton de géolocalisation introuvable");
        return;
    }
    
    // Fonction pour afficher une notification
    function showNotification(message, isError = false) {
        const notification = document.createElement('div');
        notification.className = isError ? 'geolocation-error' : 'geolocation-success';
        notification.textContent = message;
        notification.style.position = 'fixed';
        notification.style.top = '20px';
        notification.style.right = '20px';
        notification.style.padding = '10px 20px';
        notification.style.borderRadius = '5px';
        notification.style.color = 'white';
        notification.style.backgroundColor = isError ? '#e74c3c' : '#2ecc71';
        notification.style.zIndex = '9999';
        document.body.appendChild(notification);
        setTimeout(() => notification.remove(), 3000);
    }
    
    // FONCTION PRINCIPALE: Gestionnaire d'événement pour le bouton de géolocalisation
    geolocateButton.addEventListener('click', () => {
        console.log("CLIC SUR LE BOUTON DE GÉOLOCALISATION");
        
        if (!navigator.geolocation) {
            alert("Géolocalisation non supportée par votre navigateur.");
            return;
        }
        
        // Désactiver le bouton et afficher l'état de chargement
        geolocateButton.disabled = true;
        geolocateButton.innerHTML = '⏳ Localisation...';
        
        // Demander la position de l'utilisateur
        navigator.geolocation.getCurrentPosition(
            // Succès de la géolocalisation
            async (position) => {
                const userLat = position.coords.latitude;
                const userLng = position.coords.longitude;
                console.log(`POSITION UTILISATEUR: [${userLat}, ${userLng}]`);
                
                // Afficher l'info des distances
                if (distancesInfo) {
                    distancesInfo.style.display = 'block';
                }
                
                // Récupérer tous les éléments d'agriculteurs
                const listItems = document.querySelectorAll('.farmer-item');
                const gridItems = document.querySelectorAll('.grid-item');
                
                console.log(`Nombre d'éléments trouvés: ${listItems.length} liste, ${gridItems.length} grille`);
                
                if (listItems.length === 0 && gridItems.length === 0) {
                    console.error("Aucun élément d'agriculteur trouvé");
                    geolocateButton.disabled = false;
                    geolocateButton.innerHTML = '<i class="fas fa-location-crosshairs"></i> Les plus proches';
                    showNotification("Aucun agriculteur trouvé", true);
                    return;
                }
                
                // Variables pour suivre le progrès
                let successCount = 0;
                let errorCount = 0;
                let totalRequests = 0;
                let completedRequests = 0;
                const distances = [];
                
                // Fonction pour mettre à jour le bouton avec le progrès
                const updateProgress = () => {
                    const percent = totalRequests > 0 ? Math.round((completedRequests / totalRequests) * 100) : 0;
                    geolocateButton.innerHTML = `⏳ Calcul des distances... ${percent}%`;
                };
                
                // TRAITEMENT DE CHAQUE AGRICULTEUR (LISTE)
                for (const item of listItems) {
                    const id = item.getAttribute('data-id');
                    const coords = item.getAttribute('data-coordinates');
                    
                    if (!coords || !id) {
                        console.warn(`Élément incomplet - ID: ${id}, Coords: ${coords}`);
                        continue;
                    }
                    
                    // CORRECTION: Inverser l'ordre des coordonnées pour correspondre au format attendu
                    // Format attendu par l'API: "longitude,latitude"
                    const [farmLat, farmLng] = coords.split(',').map(parseFloat);
                    
                    if (isNaN(farmLat) || isNaN(farmLng)) {
                        console.warn(`Coordonnées invalides pour ID ${id}: ${coords}`);
                        continue;
                    }
                    
                    totalRequests++;
                    updateProgress();
                    
                    try {
                        // CORRECTION: Utiliser le bon ordre des coordonnées pour l'API
                        // L'API attend: start=longitude,latitude&end=longitude,latitude
                        const baseUrl = window.location.origin;
                        const url = `${baseUrl}/modules/mod_agriculteur_liste/api/get-distance.php?start=${userLng},${userLat}&end=${farmLng},${farmLat}`;
                        
                        console.log(`Appel API pour ID ${id}: ${url}`);
                        const response = await fetch(url);
                        
                        if (!response.ok) {
                            throw new Error(`Erreur HTTP: ${response.status}`);
                        }
                        
                        const data = await response.json();
                        console.log(`Réponse API pour ID ${id}:`, data);
                        
                        // Vérifier si la réponse contient une erreur
                        if (data.error) {
                            throw new Error(`Erreur API: ${data.error}`);
                        }
                        
                        // Extraire la distance de la réponse
                        if (data.features && 
                            data.features[0] && 
                            data.features[0].properties && 
                            data.features[0].properties.summary && 
                            typeof data.features[0].properties.summary.distance === 'number') {
                            
                            // Convertir de mètres en kilomètres
                            const distanceKm = data.features[0].properties.summary.distance / 1000;
                            console.log(`Distance calculée pour ID ${id}: ${distanceKm.toFixed(1)} km`);
                            
                            // CORRECTION: Sélection directe et vérification des éléments
                            const distanceElement = document.getElementById('distance-' + id);
                            if (distanceElement) {
                                console.log(`Élément distance trouvé pour ID ${id}`);
                                const distanceValue = distanceElement.querySelector('.distance-value');
                                if (distanceValue) {
                                    console.log(`Mise à jour de la distance pour ID ${id}: ${distanceKm.toFixed(1)} km`);
                                    distanceValue.textContent = distanceKm.toFixed(1);
                                    distanceValue.style.color = 'red';
                                    distanceValue.style.fontWeight = 'bold';
                                    
                                    // Rendre l'élément visible
                                    distanceElement.style.display = 'flex';
                                    distanceElement.style.backgroundColor = '#e9f5dc';
                                } else {
                                    console.error(`Élément .distance-value non trouvé pour ID ${id}`);
                                }
                            } else {
                                console.error(`Élément distance-${id} non trouvé`);
                            }
                            
                            // Mise à jour de la vue grille également
                            const gridDistanceElement = document.getElementById('grid-distance-' + id);
                            if (gridDistanceElement) {
                                console.log(`Élément grid-distance trouvé pour ID ${id}`);
                                const gridDistanceValue = gridDistanceElement.querySelector('.grid-distance-value');
                                if (gridDistanceValue) {
                                    console.log(`Mise à jour de la distance grille pour ID ${id}: ${distanceKm.toFixed(1)} km`);
                                    gridDistanceValue.textContent = distanceKm.toFixed(1);
                                    gridDistanceValue.style.color = 'red';
                                    gridDistanceValue.style.fontWeight = 'bold';
                                    
                                    // Rendre l'élément visible
                                    gridDistanceElement.style.display = 'flex';
                                    gridDistanceElement.style.backgroundColor = '#e9f5dc';
                                } else {
                                    console.error(`Élément .grid-distance-value non trouvé pour ID ${id}`);
                                }
                            }
                            
                            // Stocker pour le tri
                            distances.push({ id, distance: distanceKm, element: item });
                            successCount++;
                        } else {
                            console.error(`Structure de réponse inattendue pour ID ${id}:`, data);
                            throw new Error('Structure de réponse inattendue');
                        }
                    } catch (error) {
                        console.error(`Erreur pour ID ${id}:`, error);
                        errorCount++;
                    } finally {
                        completedRequests++;
                        updateProgress();
                        
                        // Si tous les éléments ont été traités, trier et réorganiser
                        if (completedRequests === totalRequests) {
                            finishDistanceCalculation();
                        }
                    }
                }
                
                // Fonction pour finaliser le calcul des distances
                function finishDistanceCalculation() {
                    console.log(`RÉSULTATS: ${successCount} succès, ${errorCount} erreurs`);
                    
                    // Trier les éléments par distance
                    if (distances.length > 0) {
                        console.log(`Tri des ${distances.length} éléments par distance`);
                        distances.sort((a, b) => a.distance - b.distance);
                        
                        // Réorganiser les éléments dans le DOM (vue liste)
                        const listContainer = document.getElementById('farmer-list-' + moduleId);
                        if (listContainer) {
                            console.log(`Réorganisation de la liste (container trouvé)`);
                            distances.forEach((d) => {
                                const listItem = d.element.closest('li');
                                if (listItem) {
                                    listContainer.appendChild(listItem);
                                }
                            });
                        } else {
                            console.error(`Container de liste farmer-list-${moduleId} non trouvé`);
                        }
                        
                        // Réorganiser les éléments dans le DOM (vue grille)
                        const gridContainer = document.getElementById('farmer-grid-' + moduleId);
                        if (gridContainer) {
                            console.log(`Réorganisation de la grille (container trouvé)`);
                            distances.forEach((d) => {
                                const gridItem = document.querySelector(`.grid-item[data-id="${d.id}"]`);
                                if (gridItem) {
                                    gridContainer.appendChild(gridItem);
                                }
                            });
                        } else {
                            console.log(`Container de grille farmer-grid-${moduleId} non trouvé (peut-être normal si pas en vue grille)`);
                        }
                    }
                    
                    // Réinitialiser le bouton
                    geolocateButton.disabled = false;
                    geolocateButton.innerHTML = '<i class="fas fa-location-crosshairs"></i> Les plus proches';
                    
                    // Afficher un message approprié
                    if (successCount > 0) {
                        showNotification(`${successCount} distances calculées avec succès !`);
                    } else {
                        showNotification('Impossible de calculer les distances routières.', true);
                    }
                }
            },
            // Erreur de géolocalisation
            (error) => {
                console.error("Erreur de géolocalisation:", error);
                
                let errorMessage = "Impossible d'obtenir votre position.";
                switch(error.code) {
                    case error.PERMISSION_DENIED:
                        errorMessage += " Vous avez refusé l'accès à votre position.";
                        break;
                    case error.POSITION_UNAVAILABLE:
                        errorMessage += " Les informations de position ne sont pas disponibles.";
                        break;
                    case error.TIMEOUT:
                        errorMessage += " La demande de position a expiré.";
                        break;
                }
                
                alert(errorMessage);
                showNotification(errorMessage, true);
                
                geolocateButton.disabled = false;
                geolocateButton.innerHTML = '<i class="fas fa-location-crosshairs"></i> Les plus proches';
            },
            // Options de géolocalisation
            {
                enableHighAccuracy: true,
                timeout: 10000,
                maximumAge: 0
            }
        );
    });
    
    console.log("=== INITIALISATION DE GEOLOC.JS TERMINÉE ===");
});
